<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Size;
use App\Models\Warehouse;
use App\Models\Supplier;
use App\Models\Liquid;
use App\Models\ProductSizeVariation;
use App\Models\LiquidOilVariation;

use Illuminate\Support\Str;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $products = Product::orderBy("created_at", "DESC")->get();
        $liquidproducts = Product::orderBy("created_at", "DESC")->get();

        // Create an array to store size data for each product
        $size_data = [];
        $liquid_data = [];


        // Loop through each product to retrieve its size data
        foreach ($products as $product) {
            // Retrieve size data for the current product
            $size_data[$product->id] = ProductSizeVariation::where(
                "product_id",
                $product->id
            )->get();
        }

        foreach ($liquidproducts as $product) {
            // Retrieve size data for the current product
            $liquid_data[$product->id] = LiquidOilVariation::where(
                "product_id",
                $product->id
            )->get();
        }

        return view("backend.product.index", compact("products", "size_data", "liquid_data"));
    }

    public function stockalert()
    {
        $products = Product::where("stock", "=", 0)->paginate(10);
        return view("backend.product.index")->with("products", $products);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $lq = Liquid::get();
        $size = Size::get();
        $category = Category::where("is_parent", 1)->get();
        $brand = Brand::where("is_parent", 1)->get();
        $warehouse = Warehouse::get();
        $supplier = Supplier::get();
        return view("backend.product.create")
            ->with("categories", $category)
            ->with("lqs", $lq)
            ->with("brands", $brand)
            ->with("warehouses", $warehouse)
            ->with("suppliers", $supplier)
            ->with("sizes", $size);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //return json_encode($request->color);

        $this->validate($request, [
            "title" => "string|required",
            "summary" => "string|required",
            "slug" => "unique:products,slug",
            "description" => "string|nullable",
            "photo" => "required",
            "color" => "nullable",
            "stock" => "string|nullable",
            "cat_id" => "required",
            "br_id" => "nullable",
            "is_featured" => "",
            "status" => "required|in:active,inactive",
            "offer_price" => "string|nullable",
            "cost" => "required",
            "SKU" => "nullable",
            "discount" => "nullable|numeric",
            "size_id" => "nullable",
            "unit" => "nullable",
        ]);

        $data = $request->all();
        // Generate a random string of length 5
        $randomString = Str::random(5);

        // Convert the random string to uppercase
        $uppercaseRandomString = strtoupper($randomString);

        // Concatenate the title from the request with a hyphen and the uppercase random string
        $slugSuffix = "-" . $uppercaseRandomString;

        // Generate a slug from the concatenated string
        $titleSlug = Str::slug($request->title);

        // Combine the title slug with the slug suffix
        $combinedSlug = $titleSlug . $slugSuffix;

        // Assign the combined slug to the data array
        $data["slug"] = $combinedSlug;

        $data["cost"] = $request->cost;
        $color = $request->input("colors");

        if (isset($color) && $color[0] != null && $color[1] != null) {
            $data["color"] = implode(",", $color);
            if ($data["color"][strlen($data["color"]) - 1] == ",") {
                $data["color"] = substr($data["color"], 0, -1);
            }
        }

        $cat_ids = $request->input("cat_id");

        if (isset($cat_ids)) {
            $data["cat_id"] = implode(",", $cat_ids);
        } else {
            $data["cat_id"] = "";
        }

        $br_id = $request->input("br_id");

        if (isset($br_id)) {
            $data["br_id"] = implode(",", $br_id);
        }

        if ($request->wholesale_price) {
            $data['product_type'] = 'wholesale';
        } else {
            $data['product_type'] = 'normal';
        }

        $data["is_featured"] = $request->input("is_featured", 0);
        //dd($data);
        $status = Product::create($data);

        $size_ids = $request->input("size_id");


        if (!empty($size_ids)) {
            foreach ($size_ids as $index => $size_id) {
                // Validate the data for each size variation before creating
                $request->validate([
                    "size_id.$index" => "nullable",
                    "size_price.$index" => "nullable",
                    "size_stock.$index" => "nullable",
                ]);

                // Create the size variation associated with the product
                $status->product_size_variations()->create([
                    "size_id" => $size_id,
                    "price" => $request->size_price[$index],
                    "stock" => $request->size_stock[$index],
                ]);
            }
        }
        $liquidInput = (array) $request->input("liquid");

        if (!empty($liquidInput)) {
            // Retrieve liquid variations data from the request and create records
            foreach ($request->liquid as $index => $variationData) {
                $status->variations()->create([
                    "liquid_id" => $liquidInput[$index],
                    "price" => $request->liquid_price[$index],
                    "stock" => $request->liquid_stock[$index],
                ]);
            }
        }
        if ($status) {
            request()
                ->session()
                ->flash("success", "Product Successfully Create");
        } else {
            request()
                ->session()
                ->flash("error", "Please try again!!");
        }
        return redirect()->route("product.index");
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        // Get all sizes
        $sizes = Size::all();

        // Find the product
        $product = Product::findOrFail($id);

        // Get size variations for the current product
        $sizev = ProductSizeVariation::where('product_id', $product->id)->get();

        // Get categories and brands
        $category = Category::where("is_parent", 1)->get();
        $brand = Brand::where("is_parent", 1)->get();
        $warehouse = Warehouse::get();
        $supplier = Supplier::get();

        return view("backend.product.edit")
            ->with("product", $product)
            ->with("categories", $category)
            ->with("brands", $brand)
            ->with("warehouses", $warehouse)
            ->with("suppliers", $supplier)
            ->with("sizevs", $sizev)
            ->with("sizes", $sizes); // Pass all sizes to the view
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $product = Product::findOrFail($id);

        $validatedData = $request->validate([
            "title" => "required|string",
            "summary" => "required|string",
            "slug" => "required",
            "description" => "nullable|string",
            "photo" => "required",
            "size" => "nullable",
            "color" => "nullable|array",
            "stock" => "required|numeric",
            "cat_id" => "required|array",
            "br_id" => "nullable|array",
            "is_featured" => "nullable|boolean",
            "status" => "required|in:active,inactive",
            "offer_price" => "nullable",
            "cost" => "required",
            "discount" => "nullable|numeric",
        ]);

        $data = $request->all();

        // Handle color, br_id, and cat_id as comma-separated strings
        $data["color"] = isset($data["color"])
            ? implode(",", $data["color"])
            : null;
        $data["br_id"] = isset($data["br_id"])
            ? implode(",", $data["br_id"])
            : null;
        $data["cat_id"] = isset($data["cat_id"])
            ? implode(",", $data["cat_id"])
            : null;

        // Update is_featured field
        $data["is_featured"] = $request->input("is_featured", 0);

        // Update the product record
        $status = $product->fill($data)->save();

        // Handle size variations
        $size_ids = $request->input("size_id");
        if (isset($size_ids)) {
            foreach ($request->size_id as $index => $size_id) {
                $validatedSizeData = $request->validate([
                    "size_id.$index" => "nullable",
                    "size_price.$index" => "nullable",
                    "size_stock.$index" => "nullable",
                ]);

                // Create or update the size variation associated with the product
                $product->product_size_variations()->updateOrCreate(
                    ["size_id" => $size_id],
                    [
                        "price" => $request->size_price[$index],
                        "stock" => $request->size_stock[$index],
                    ]
                );
            }
        }

        if ($status) {
            return redirect()
                ->route("product.index")
                ->with("success", "Product successfully updated");
        } else {
            return redirect()
                ->route("product.index")
                ->with("error", "Failed to update product");
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $product = Product::findOrFail($id);
        $status = $product->delete();

        if ($status) {
            request()
                ->session()
                ->flash("success", "Product successfully deleted");
        } else {
            request()
                ->session()
                ->flash("error", "Error delete product");
        }
        return redirect()->route("product.index");
    }

    public function delete($id)
    {
        try {
            $sizeVariation = ProductSizeVariation::findOrFail($id);
            $sizeVariation->delete();

            return back()->with('success', 'Size variation deleted successfully');
        } catch (\Exception $e) {
            return back()->with('error', 'Error deleting size variation');
        }
    }


    public function active($id)
    {
        $product = Product::findOrFail($id);
        $product->is_featured = 1;
        $status = $product->save();

        if ($status) {
            request()
                ->session()
                ->flash("success", "Product successfully Active!");
        } else {
            request()
                ->session()
                ->flash("error", "Error while active product");
        }
        return redirect()->route("product.index");
    }
    public function deactive($id)
    {
        $product = Product::findOrFail($id);
        $product->is_featured = 0;
        $status = $product->save();

        if ($status) {
            request()
                ->session()
                ->flash("success", "Product successfully Deactive!");
        } else {
            request()
                ->session()
                ->flash("error", "Error while deactive product");
        }
        return redirect()->route("product.index");
    }
}
